<?php
/**
 * LitePay Non-Merchant Gateway for Blesta
 *
 * @package blesta
 * @subpackage blesta.components.gateways.litepay
 * @copyright Copyright (c) 2026 LitePay.ch
 * @link http://www.litepay.ch/ LitePay
 */
class Litepay extends NonmerchantGateway
{
    /**
     * @var array An array of meta data for this gateway
     */
    private $meta;

    /**
     * @var string The merchant ID of Phillips Data Inc.
     */
    private $merchant_id = '';

    /**
     * Construct a new merchant gateway
     */
    public function __construct()
    {
        $this->loadConfig(dirname(__FILE__) . DS . 'config.json');

        // Load components required by this gateway
        Loader::loadComponents($this, ['Input']);

        // Load the language required by this gateway
        Language::loadLang('litepay', null, dirname(__FILE__) . DS . 'language' . DS);
    }

    /**
     * Sets the currency code to be used for all subsequent payments
     *
     * @param string $currency The ISO 4217 currency code to be used for subsequent payments
     */
    public function setCurrency($currency)
    {
        $this->currency = $currency;
    }

    /**
     * Create and return the view content required to modify the settings of this gateway
     *
     * @param array $meta An array of meta (settings) data belonging to this gateway
     * @return string HTML content containing the fields to update the meta data for this gateway
     */
    public function getSettings(array $meta = null)
    {
        $this->view = $this->makeView('settings', 'default', str_replace(ROOTWEBDIR, '', dirname(__FILE__) . DS));

        // Load the helpers required for this view
        Loader::loadHelpers($this, ['Form', 'Html']);

        $this->view->set('meta', $meta);

        return $this->view->fetch();
    }

    /**
     * Validates the given meta (settings) data to be updated for this gateway
     *
     * @param array $meta An array of meta (settings) data to be updated for this gateway
     * @return array The meta data to be updated in the database for this gateway, or reset into the form on failure
     */
    public function editSettings(array $meta)
    {
        // Verify meta data is valid
        $rules = [];

        $this->Input->setRules($rules);

        // Validate the given meta data to ensure it meets the requirements
        $this->Input->validates($meta);
        // Return the meta data, no changes required regardless of success or failure for this gateway
        return $meta;
    }

    /**
     * Returns an array of all fields to encrypt when storing in the database
     *
     * @return array An array of the field names to encrypt when storing in the database
     */
    public function encryptableFields()
    {
        return ['vendor', 'secret'];
    }

    /**
     * Sets the meta data for this particular gateway
     *
     * @param array $meta An array of meta data to set for this gateway
     */
    public function setMeta(array $meta = null)
    {
        $this->meta = $meta;
    }

    /**
     * Returns all HTML markup required to render an authorization and capture payment form
     *
     * @param array $contact_info An array of contact info including:
     *  - id The contact ID
     *  - client_id The ID of the client this contact belongs to
     *  - user_id The user ID this contact belongs to (if any)
     *  - contact_type The type of contact
     *  - contact_type_id The ID of the contact type
     *  - first_name The first name on the contact
     *  - last_name The last name on the contact
     *  - title The title of the contact
     *  - company The company name of the contact
     *  - address1 The address 1 line of the contact
     *  - address2 The address 2 line of the contact
     *  - city The city of the contact
     *  - state An array of state info including:
     *      - code The 2 or 3-character state code
     *      - name The local name of the country
     *  - country An array of country info including:
     *      - alpha2 The 2-character country code
     *      - alpha3 The 3-cahracter country code
     *      - name The english name of the country
     *      - alt_name The local name of the country
     *  - zip The zip/postal code of the contact
     * @param float $amount The amount to charge this contact
     * @param array $invoice_amounts An array of invoices, each containing:
     *  - id The ID of the invoice being processed
     *  - amount The amount being processed for this invoice (which is included in $amount)
     * @param array $options An array of options including:
     *  - description The Description of the charge
     *  - return_url The URL to redirect users to after a successful payment
     *  - recur An array of recurring info including:
     *      - start_date The date/time in UTC that the recurring payment begins
     *      - amount The amount to recur
     *      - term The term to recur
     *      - period The recurring period (day, week, month, year, onetime) used in conjunction with term in
     *        order to determine the next recurring payment
     * @return mixed A string of HTML markup required to render an authorization and capture payment form, or an
     *  array of HTML markup
     */
    public function buildProcess(array $contact_info, $amount, array $invoice_amounts = null, array $options = null)
    {
        // Force 8-decimal places only
        $amount = round($amount, 8);
        if (isset($options['recur']['amount'])) {
            $options['recur']['amount'] = round($options['recur']['amount'], 8);
        }

        $post_to = 'https://litepay.ch/p/';

        // An array of key/value hidden fields to set for the payment form
        $fields = [ 
            'vendor' => $this->meta['vendor'],
            'secret' => $this->meta['secret'],
            'currency' => $this->currency,
            'price' => $amount,
            'invoice' => $invoice_amounts[0]['id'],
            'callbackUrl' => Configure::get('Blesta.gw_callback_url')
                . Configure::get('Blesta.company_id')
                . '/litepay/?client_id='
                . (isset($contact_info['client_id']) ? $contact_info['client_id'] : null)
                . '&secret=' . (isset($this->meta['secret']) ? $this->meta['secret'] : null)
                . '&amount=' . $amount
                . '&invoice=' . $invoice_amounts[0]['id']
                . '&currency=' . $this->currency,
            'returnUrl' => (isset($options['return_url']) ? $options['return_url'] : null),
            'email' => (isset($contact_info['email']) ? $contact_info['email'] : null),
            'redirect' => 'true'
        ];

        $regular_btn = $this->buildForm($post_to, $fields, false);
        return $regular_btn;
    }

    /**
     * Builds the HTML form
     *
     * @param string $post_to The URL to post to
     * @param array $fields An array of key/value input fields to set in the form
     * @param boolean $recurring True if this is a recurring payment request, false otherwise
     * @return string The HTML form
     */
    private function buildForm($post_to, $fields, $recurring = false)
    {
        $this->view = $this->makeView('process', 'default', str_replace(ROOTWEBDIR, '', dirname(__FILE__) . DS));

        // Load the helpers required for this view
        Loader::loadHelpers($this, ['Form', 'Html']);

        $this->view->set('post_to', $post_to);
        $this->view->set('fields', $fields);
        $this->view->set('recurring', $recurring);

        return $this->view->fetch();
    }

    private function checkIpnRequestIsValid($get)
    {
        $error_msg = null;

        // Verify secret
        if (isset($get['secret']) && !empty($get['secret'])) {
            if (trim((isset($this->meta['secret']) ? $this->meta['secret'] : null)) != trim($get['secret'])) {
                $error_msg = 'Invalid secret';
            }
        } else {
            $error_msg = 'No IPN secret passed';
        }
        

        // Verify invoice parameter exists and is valid
        if (!isset($get['invoice']) || empty($get['invoice'])) {
            $error_msg = 'Missing invoice parameter';
        }

        if (!isset($get['client_id']) || empty($get['client_id'])) {
            $error_msg = 'Missing client_id parameter';
        }
        
        // Load the Invoices model to validate invoice ID
        Loader::loadModels($this, ['Invoices']);
        $invoice = $this->Invoices->get($get['invoice']);
        
        if (!$invoice) {
            $error_msg = 'Invalid invoice ID';
        }

        // Verify amount parameter exists and is valid
        if (!isset($get['amount']) || !is_numeric($get['amount']) || $get['amount'] <= 0) {
            $error_msg = 'Invalid or missing amount parameter';
        }

        // Verify amount matches invoice amount due
        // $amount_due = $invoice->due;
        // if (abs((float)$get['amount'] - (float)$amount_due) > 0.00000001) {
        //     $error_msg = 'Amount does not match invoice amount due';
        // }

        // Verify currency parameter exists
        if (!isset($get['currency']) || empty($get['currency'])) {
            $error_msg = 'Missing currency parameter';
        }

        // Verify currency matches invoice currency
        if (strtoupper($get['currency']) != strtoupper($invoice->currency)) {
            $error_msg = 'Currency does not match invoice currency';
        }

        // Verify value parameter (must be integer or numeric)
        if (!isset($get['value']) || !is_numeric($get['value'])) {
            $error_msg = 'Invalid or missing value parameter (must be numeric)';
        }

        // Verify coin parameter (must be text only, no special characters)
        if (!isset($get['coin']) || empty($get['coin'])) {
            $error_msg = 'Missing coin parameter';
        }
        if (!preg_match('/^[a-zA-Z0-9]+$/', $get['coin'])) {
            $error_msg = 'Invalid coin parameter (must be alphanumeric text only)';
        }

        // Verify input_address parameter exists
        if (!isset($get['input_address']) || empty($get['input_address'])) {
            $error_msg = 'Missing input_address parameter';
        }

        // Verify destination_address parameter exists
        if (!isset($get['destination_address']) || empty($get['destination_address'])) {
            $error_msg = 'Missing destination_address parameter';
        }

        // Verify transaction_hash parameter exists
        if (!isset($get['transaction_hash']) || empty($get['transaction_hash'])) {
            $error_msg = 'Missing transaction_hash parameter';
        }

        // Verify confirmations parameter (must be integer)
        if (!isset($get['confirmations'])) {
            $error_msg = 'Missing confirmations parameter';
        }
        if (!is_numeric($get['confirmations']) || (int)$get['confirmations'] != $get['confirmations']) {
            $error_msg = 'Invalid confirmations parameter (must be an integer)';
        }

        // All parameters are valid
        if ($error_msg) {
            return $error_msg;
        }
    }

    /**
     * Validates the incoming POST/GET response from the gateway to ensure it is
     * legitimate and can be trusted.
     *
     * @param array $get The GET data for this request
     * @param array $post The POST data for this request
     * @return array An array of transaction data, sets any errors using Input if the data fails to validate
     *  - client_id The ID of the client that attempted the payment
     *  - amount The amount of the payment
     *  - currency The currency of the payment
     *  - invoices An array of invoices and the amount the payment should be applied to (if any) including:
     *      - id The ID of the invoice to apply to
     *      - amount The amount to apply to the invoice
     *  - status The status of the transaction (approved, declined, void, pending, reconciled, refunded, returned)
     *  - reference_id The reference ID for gateway-only use with this transaction (optional)
     *  - transaction_id The ID returned by the gateway to identify this transaction
     *  - parent_transaction_id The ID returned by the gateway to identify this transaction's original
     *    transaction (in the case of refunds)
     */
    public function validate(array $get, array $post)
    {
        // Log request received
        $this->log((isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : null), serialize($get), 'output', true);

        // Ensure IPN is verified, and validate that the merchant ID is correct
        $error_msg = $this->checkIpnRequestIsValid($get);

        if ($error_msg) {
            $report = 'IPN Error: ' . $error_msg . "\n\n";
            $report .= "GET Variables\n\n";
            foreach ($get as $k => $v) {
                $report .= '|' . $k . '| = |' . $v . "|\n";
            }
            $this->log((isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : null), serialize($report), 'output', true);
            exit();
        }

        $transaction_hash = (isset($get['transaction_hash']) ? $get['transaction_hash']."-".(isset($get['client_id']) ? $get['client_id'] : null) : null);

        $this->success($get, $post);
        return  [
            'client_id' => (isset($get['client_id']) ? $get['client_id'] : null),
            'amount' => (isset($get['amount']) ? $get['amount'] : null),
            'currency' => (isset($get['currency']) ? $get['currency'] : null),
            'invoices' => [
                'id' => (isset($get['invoice']) ? $get['invoice'] : null),
                'amount' => (isset($get['amount']) ? $get['amount'] : null),
             ],
            'reference_id' => null,
            'parent_transaction_id' => '',
            'transaction_id' => $transaction_hash,
            'status' => 'approved'
        ];

    }

    /**
     * Returns data regarding a success transaction. This method is invoked when
     * a client returns from the non-merchant gateway's web site back to Blesta.
     *
     * @param array $get The GET data for this request
     * @param array $post The POST data for this request
     * @return array An array of transaction data, may set errors using Input if the data appears invalid
     *  - client_id The ID of the client that attempted the payment
     *  - amount The amount of the payment
     *  - currency The currency of the payment
     *  - invoices An array of invoices and the amount the payment should be applied to (if any) including:
     *      - id The ID of the invoice to apply to
     *      - amount The amount to apply to the invoice
     *  - status The status of the transaction (approved, declined, void, pending, reconciled, refunded, returned)
     *  - transaction_id The ID returned by the gateway to identify this transaction
     *  - parent_transaction_id The ID returned by the gateway to identify this transaction's original transaction
     */
    public function success(array $get, array $post)
    {
        print("*ok*");
    }

    /**
     * Serializes an array of invoice info into a string
     *
     * @param array A numerically indexed array invoices info including:
     *  - id The ID of the invoice
     *  - amount The amount relating to the invoice
     * @return string A serialized string of invoice info in the format of key1=value1|key2=value2
     */
    private function serializeInvoices(array $invoices)
    {
        $str = '';
        foreach ($invoices as $i => $invoice) {
            $str .= ($i > 0 ? '|' : '') . $invoice['id'] . '=' . $invoice['amount'];
        }
        return $str;
    }

    /**
     * Unserializes a string of invoice info into an array
     *
     * @param string A serialized string of invoice info in the format of key1=value1|key2=value2
     * @return array A numerically indexed array invoices info including:
     *  - id The ID of the invoice
     *  - amount The amount relating to the invoice
     */
    private function unserializeInvoices($str)
    {
        $invoices = [];
        $temp = explode('|', $str);
        foreach ($temp as $pair) {
            $pairs = explode('=', $pair, 2);
            if (count($pairs) != 2) {
                continue;
            }
            $invoices[] = ['id' => $pairs[0], 'amount' => $pairs[1]];
        }
        return $invoices;
    }
}
